<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\Cart;
use App\Models\State;
use App\Models\City;
use App\Models\Order;
use App\Models\Coupon;
use App\Models\CouponUsage;
use App\Models\Address;
use App\Models\Carrier;
use App\Models\CombinedOrder;
use App\Models\Country;
use App\Models\Pincode;
use App\Models\Product;
use App\Models\User;
use App\Utility\EmailUtility;
use App\Utility\NotificationUtility;
use Session;
use Auth;
use Hash;
use Illuminate\Support\Facades\Validator;
use Mail;
use Illuminate\Support\Facades\Log;
use App\Mail\InvoiceEmailManager;
use App\Http\Controllers\Api\V2\PhonepeController as ApiPhonepeController;
use Exception;
use Illuminate\Support\Facades\DB;

class CheckoutController extends Controller
{
    public function __construct()
    {
        //
    }

    public function index(Request $request)
    {
        if (get_setting('guest_checkout_activation') == 0 && auth()->user() == null) {
            return redirect()->route('user.login');
        }

        if (auth()->check() && !$request->user()->hasVerifiedEmail()) {
            return redirect()->route('verification.notice');
        }

        $address_id = 0;
        $countries = [];
        $states = [];
        $cities = [];
        $pincodes = []; // Add pincodes variable
        $address = null;

        if (auth()->check()) {
            $user_id = Auth::user()->id;
            $carts = Cart::where('user_id', $user_id)->active()->get();
            $address = Address::where('user_id', $user_id)->where('set_default', 1)->first();
            if (!$address) {
                $address = Address::where('user_id', $user_id)->first();
            }
            if ($address) {
                $address_id = $address->id;
                $countries = Country::where('status', 1)->pluck('name', 'id');
                if ($address->country_id) {
                    $states = State::where('country_id', $address->country_id)->where('status', 1)->pluck('name', 'id');
                }
                if ($address->state_id) {
                    $cities = City::where('state_id', $address->state_id)->where('status', 1)->pluck('name', 'id');
                }
                if ($address->city_id) {
                    $pincodes = Pincode::where('city_id', $address->city_id)->where('is_active', 1)->pluck('pincode');
                }
            }
        } else {
            $temp_user_id = $request->session()->get('temp_user_id');
            $carts = ($temp_user_id != null) ? Cart::where('temp_user_id', $temp_user_id)->active()->get() : [];
        }

        if ($carts->isEmpty()) {
            return redirect('/')->with('error', 'Please Select cart items to Proceed');
        }

        if ($address) {
            $shipping_info = [
                'country_id'    => $address->country_id,
                'state_id'      => $address->state_id,
                'city_id'       => $address->city_id
            ];
            $default_shipping_type = 'home_delivery';

            if ($carts && count($carts) > 0) {
                $carts->toQuery()->update(['address_id' => $address->id]);
                $carts = $carts->fresh();

                $carrier_list = array();
                if (get_setting('shipping_type') == 'carrier_wise_shipping') {
                    $default_shipping_type = 'carrier';
                    $country = Country::find($address->country_id);
                    $zone = $country ? $country->zone_id : 0;

                    $carrier = Carrier::where('status', 1)
                        // ->whereIn('id', function ($query) use ($zone) {
                        //     $query->select('carrier_id')->from('carrier_range_prices')
                        //         ->where('zone_id', $zone);
                        // })
                        // ->orWhere('free_shipping', 1)
                        ->first();

                    if ($carrier) {
                        $shipping_info['carrier_id'] = $carrier->id;
                    }
                }

                foreach ($carts as $key => $cart) {
                    $product = $cart->product;
                    $product_stock = $product->stocks->where('variant', $cart->variation)->first();
                    $quantity = $product_stock->qty;
                    $cart->shipping_cost = get_product_shipping_cost($product, $quantity, $cart->variation, $carts, $shipping_info);
                    $cart->shipping_type = $default_shipping_type;
                    $cart->carrier_id = isset($shipping_info['carrier_id']) ? $shipping_info['carrier_id'] : null;
                    $cart->save();
                }
                $carts = $carts->fresh();
            }
        }

        if ($carts->count() > 0) {
            return view('frontend.checkout-new', compact('carts', 'pincodes', 'cities', 'states', 'address', 'countries'));
        }
        // flash(translate('Please Select cart items to Proceed'))->error();
        // return redirect()->back();
    }

    public function countryfetch(Request $request)
    {
        $text = $request->input('text');

        $list = Country::where('name', 'like', $text . '%')->where('status', 1)->orderBy('name')->get();

        return response()->json($list);
    }

    public function statefetch(Request $request)
    {
        $text = $request->input('text');
        $countrycode = $request->input('countrycode');

        $list = State::where('name', 'like', $text . '%')->where('country_id', $countrycode)->where('status', 1)->orderBy('name')->get();

        return response()->json($list);
    }

    public function cityfetch(Request $request)
    {
        $text = $request->input('text');
        $statecode = $request->input('statecode');

        $list = City::where('name', 'like', $text . '%')->where('state_id', $statecode)->where('status', 1)->orderBy('name')->get();

        return response()->json($list);
    }

    public function pinfetch(Request $request)
    {
        $citycode = $request->input('citycode');

        $list = Pincode::where('city_id', $citycode)->where('is_active', 1)->orderBy('pincode')->get();

        return response()->json($list);
    }

    public function get_shipping_attribute_data(Request $request, $type, $parent_id = null)
    {
        $data = [];
        $html = "";
        $dtype = "";

        if ($type === 'country') {
            $data = get_active_countries($parent_id, ['id', 'name']);
            $dtype = 'Country';
        } elseif ($type === 'state') {
            $data = get_active_states($parent_id);
            $dtype = 'State';
        } elseif ($type === 'city') {
            $data = get_active_cities($parent_id);
            $dtype = 'City';
        } elseif ($type === 'pincode') {
            $data = get_active_pincodes($parent_id);
            $dtype = 'Pincode';
        }

        if (count($data) > 1) {
            return response()->json([
                'success' => true,
                'data' => $data,
            ]);
        } else {
            return response()->json([
                'success' => false,
                'data' => '',
            ]);
        }

        if (count($data)) {
            $html .= '<option value="">' . translate("Select " . $dtype) . '</option>';

            foreach ($data as $item) {
                $value = $dtype === 'Pincode' ? $item->pincode : $item->name;
                $html .= '<option value="' . $item->id . '">' . $value . '</option>';
            }

            $shipping_info = [
                'country_id' => $request->country_id,
                'state_id'   => $request->state_id,
                'city_id'    => $request->city_id,
                'pincode'    => $request->pincode,
                'address'    => $request->address,
            ];

            $carts = Cart::with('product')->where('user_id', Auth::id())->active()->get();
            foreach ($carts as $cart) {
                $product = $cart->product;
                $product_stock = $product->stocks->where('variant', $cart->variation)->first();
                $quantity = $product_stock->qty;
                $cart->shipping_cost = get_product_shipping_cost($product, $quantity, $cart->variation, $carts, $shipping_info);
                $cart->save();
            }

            $carts = $carts->fresh();

            return response()->json([
                'success'             => true,
                'html'                => $html,
                'dtype'               => $dtype,
                'order_summary_view'  => view('frontend.partials.cart.order-summary', [
                    'carts' => $carts
                ])->render()
            ]);
        }

        return response()->json([
            'success' => false,
            'html'    => ""
        ]);
    }

    public function checkout(Request $request)
    {

        // echo $request->get('country') . '/' . $request->get('state') . '/' . $request->get('city') . '/' . $request->input('postal_code');
        // exit;
        try {
            if (get_setting('guest_checkout_activation') == 0 && auth()->user() == null) {
                return redirect()->route('user.login');
            }

            if (auth()->check() && !$request->user()->hasVerifiedEmail()) {
                return redirect()->route('verification.notice');
            }

            if ($request->payment_option == null) {
                flash(translate('There is no payment option selected.'))->warning();
                return redirect()->route('checkout')->withInput($request->all());
            }

            if (auth()->user() == null) {
                $request->put('country_id', $request->get('country'));
                $request->put('state_id', $request->get('state'));
                $request->put('city_id', $request->get('city'));

                $guest_user = $this->createUser($request->except('_token', 'payment_option'));
                if (gettype($guest_user) == "object") {
                    $errors = $guest_user;
                    return redirect()->route('checkout')->withErrors($errors)->withInput($request->all());
                }

                if ($guest_user == 0) {
                    flash(translate('Please try again later.'))->warning();
                    return redirect()->route('checkout')->withInput($request->all());
                }
            }

            $user = auth()->user();
            $carts = Cart::with('product')->where('user_id', $user->id)->active()->get();
            Log::info('Cart Items Before Checkout: ' . json_encode($carts->toArray()));

            if ($carts->isEmpty()) {
                flash(translate('Your cart is empty'))->warning();
                return redirect()->route('home');
            }

            if (get_setting('minimum_order_amount_check') == 1) {
                $subtotal = 0;
                foreach ($carts as $key => $cartItem) {
                    $product_stock = $cartItem->product->stocks->where('variant', $cartItem->variation)->first();
                    if ($product_stock) {
                        $subtotal += floatval($product_stock->price) * $cartItem->quantity;
                    }
                }
                if ($subtotal < get_setting('minimum_order_amount')) {
                    flash(translate('Your order amount is less than the minimum order amount'))->warning();
                    return redirect()->route('home');
                }
            }

            // Create the order with a "pending" status
            $orderController = new OrderController();
            $combined_order = $orderController->store($request, $carts);

            if (!$combined_order) {
                Log::error('Combined Order ID not found or invalid after store: ' . $combined_order_id);
                flash(translate('Failed to create order.'))->error();
                return redirect()->route('checkout');
            }

            // Set payment type and data
            $request->session()->put('payment_type', 'cart_payment');
            $data = [
                'combined_order_id' => $combined_order->id,
                'payment_method' => $request->payment_option,
                'user_id' => $user->id,
                'amount' => $combined_order->grand_total,
            ];
            $request->session()->put('payment_data', $data);

            if ($combined_order) {
                if ($request->payment_option == 'cash_on_delivery') {
                    // For COD, confirm the order immediately
                    if (count($carts) > 0) {
                        $carts->toQuery()->delete();
                    }
                    flash(translate('Your order has been placed successfully.'))->success();
                    return redirect()->route('order_confirmed');
                } else if ($request->payment_option == 'wallet') {
                    if ($user->balance < $combined_order->grand_total) {
                        return redirect()->route('checkout')->with('error', translate('Insufficient wallet balance.'));
                    }
                    $user->balance -= $combined_order->grand_total;
                    $user->save();
                    $order_details = Order::withoutGlobalScope('enabled')
                        ->where('combined_order_id', $combined_order->id)
                        ->get();

                    // Update order status to paid
                    foreach ($order_details as $item) {
                        $item->payment_status = 'paid';
                        $item->view_status = 1;
                        $item->save();
                    }

                    if (count($carts) > 0) {
                        $carts->toQuery()->delete();
                    }
                    flash(translate('Your order has been placed successfully.'))->success();
                    return redirect()->route('order_confirmed');
                }

                // For non-COD methods, redirect to payment gateway without confirming the order
                if ($request->payment_option == 'phonepe') {
                    $phonepeController = new ApiPhonepeController();
                    $request->merge([
                        'payment_type' => 'cart_payment',
                        'combined_order_id' => $combined_order->id
                    ]);
                    Log::info('Checkout - Data Before PhonePe Pay: ' . json_encode($request->all()));
                    return $phonepeController->pay($request); // Payment gateway will handle redirection
                }

                $decorator = __NAMESPACE__ . '\\Payment\\' . str_replace(' ', '', ucwords(str_replace('_', ' ', $request->payment_option))) . "Controller";
                if (class_exists($decorator)) {
                    return (new $decorator)->pay($request); // Payment gateway will handle redirection
                } else {
                    // Manual payment handling
                    $manual_payment_data = [
                        'name' => $request->payment_option,
                        'amount' => $combined_order->grand_total,
                        'trx_id' => $request->trx_id,
                        'photo' => $request->photo
                    ];
                    foreach ($combined_order->orders as $order) {
                        $order->manual_payment = 1;
                        $order->manual_payment_data = json_encode($manual_payment_data);
                        $order->save();
                    }
                    flash(translate('Your order has been placed. Please complete the payment.'))->success();
                    return redirect()->route('order_confirmed');
                }
            }

            flash(translate('An error occurred during checkout.'))->error();
            return redirect()->route('checkout')->withInput($request->all());
        } catch (\Exception $e) {
            Log::error('Checkout submition error: ' . $e->getMessage() . ' On Line : ' . $e->getLine());
            flash(translate('An error occurred during checkout.'))->error();
            return redirect()->route('checkout')->withInput($request->all());
        }
    }

    public function createUser($guest_shipping_info)
    {
        $validator = Validator::make($guest_shipping_info, [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users|max:255',
            'phone' => 'required|max:12',
            'address' => 'required|max:255',
            'country_id' => 'required|Integer',
            'state_id' => 'required|Integer',
            'city_id' => 'required|Integer',
            'postal_code' => 'nullable' // This is pincode
        ]);

        if ($validator->fails()) {
            return $validator->errors();
        }

        // Verify the selected pincode is active for the city
        $pincode_exists = Pincode::where('city_id', $guest_shipping_info['city_id'])
            ->where('pincode', $guest_shipping_info['postal_code'])
            ->where('is_active', 1)
            ->exists();
        if (!$pincode_exists) {
            $errors = new \Illuminate\Support\MessageBag();
            $errors->add('postal_code', 'The selected pincode is not active or not available for this city.');
            return $errors;
        }

        $success = 1;
        $password = substr(hash('sha512', rand()), 0, 8);
        $isEmailVerificationEnabled = get_setting('email_verification');

        $user = new User();
        $user->name = $guest_shipping_info['name'];
        $user->email = $guest_shipping_info['email'];
        $user->phone = addon_is_activated('otp_system') ? '+' . $guest_shipping_info['country_code'] . $guest_shipping_info['phone'] : null;
        $user->password = Hash::make($password);
        $user->email_verified_at = $isEmailVerificationEnabled != 1 ? date('Y-m-d H:m:s') : null;
        $user->save();

        try {
            EmailUtility::customer_registration_email('registration_from_system_email_to_customer', $user, $password);
        } catch (\Exception $e) {
            $success = 0;
            $user->delete();
        }

        if ($success == 0) {
            return $success;
        }

        if ($isEmailVerificationEnabled == 1) {
            EmailUtility::email_verification($user, 'customer');
        }

        if ((get_email_template_data('customer_reg_email_to_admin', 'status') == 1)) {
            try {
                EmailUtility::customer_registration_email('customer_reg_email_to_admin', $user, null);
            } catch (\Exception $e) {
            }
        }

        $address = new Address;
        $address->user_id = $user->id;
        $address->address = $guest_shipping_info['address'];
        $address->country_id = $guest_shipping_info['country_id'];
        $address->state_id = $guest_shipping_info['state_id'];
        $address->city_id = $guest_shipping_info['city_id'];
        $address->postal_code = $guest_shipping_info['postal_code']; // This is pincode
        $address->phone = '+' . $guest_shipping_info['country_code'] . $guest_shipping_info['phone'];
        $address->longitude = isset($guest_shipping_info['longitude']) ? $guest_shipping_info['longitude'] : null;
        $address->latitude = isset($guest_shipping_info['latitude']) ? $guest_shipping_info['latitude'] : null;
        $address->save();

        $carts = Cart::where('temp_user_id', session('temp_user_id'))->get();
        $carts->toQuery()->update([
            'user_id' => $user->id,
            'temp_user_id' => null
        ]);
        $carts->toQuery()->active()->update([
            'address_id' => $address->id
        ]);

        auth()->login($user);

        Session::forget('temp_user_id');
        Session::forget('guest_shipping_info');

        return $success;
    }

    public function checkout_done($combined_order_id, $payment)
    {
        $combined_order = CombinedOrder::findOrFail($combined_order_id);

        foreach ($combined_order->orders as $order) {
            $order = Order::findOrFail($order->id);
            $order->payment_status = 'paid';
            $order->payment_details = $payment;
            $order->save();

            EmailUtility::order_email($order, 'paid');
            calculateCommissionAffilationClubPoint($order);

            if ($order->payment_type == 'phonepe') {
                $array = [
                    'view' => 'emails.invoice',
                    'subject' => translate('Order Confirmation - Order Code: ') . $order->code,
                    'from' => env('MAIL_FROM_ADDRESS'),
                    'order' => $order
                ];
                Mail::to($order->user->email)->send(new InvoiceEmailManager($array));
            }
        }

        // Delete cart items here after successful payment
        $user = auth()->user();
        $carts = Cart::where('user_id', $user->id)->active()->get();
        if (count($carts) > 0) {
            $carts->toQuery()->delete();
            Log::info('Cart deleted after successful payment for Combined Order ID: ' . $combined_order_id);
        }

        Session::put('combined_order_id', $combined_order_id);
        flash(translate('Payment successful! Your order has been placed.'))->success();
        return redirect()->route('order_confirmed');
    }

    public function get_shipping_info(Request $request)
    {
        if (get_setting('guest_checkout_activation') == 0 && auth()->user() == null) {
            return redirect()->route('user.login');
        }

        if (auth()->user() != null) {
            $user_id = Auth::user()->id;
            $carts = Cart::where('user_id', $user_id)->get();
        } else {
            $temp_user_id = $request->session()->get('temp_user_id');
            $carts = ($temp_user_id != null) ? Cart::where('temp_user_id', $temp_user_id)->get() : [];
        }
        if ($carts && count($carts) > 0) {
            $categories = Category::all();
            return view('frontend.shipping_info', compact('categories', 'carts'));
        }
        flash(translate('Your cart is empty'))->success();
        return back();
    }

    public function store_shipping_info(Request $request)
    {
        $auth_user = auth()->user();
        $temp_user_id = $request->session()->has('temp_user_id') ? $request->session()->get('temp_user_id') : null;

        if ($auth_user == null && get_setting('guest_checkout_activation') == 0) {
            return redirect()->route('user.login');
        }

        if ($auth_user != null) {
            if ($request->address_id == null) {
                flash(translate("Please add shipping address"))->warning();
                return redirect()->route('checkout.shipping_info');
            }

            $carts = Cart::where('user_id', $auth_user->id)->get();
            foreach ($carts as $key => $cartItem) {
                $cartItem->address_id = $request->address_id;
                $cartItem->save();
            }
        } else {
            if (get_setting('guest_checkout_activation') == 1) {
                if (
                    $request->name == null || $request->email == null || $request->address == null ||
                    $request->country_id == null || $request->state_id == null || $request->city_id == null ||
                    $request->postal_code == null || $request->phone == null
                ) {
                    flash(translate("Please add shipping address"))->warning();
                    return redirect()->route('checkout.shipping_info');
                }

                // Verify city has active pincodes and the selected pincode is valid
                $city_has_active_pincodes = Pincode::where('city_id', $request->city_id)
                    ->where('pincode', $request->postal_code)
                    ->where('is_active', 1)
                    ->exists();
                if (!$city_has_active_pincodes) {
                    flash(translate("The selected city or pincode is not available for shipping."))->warning();
                    return redirect()->route('checkout.shipping_info');
                }

                $shipping_info['name'] = $request->name;
                $shipping_info['email'] = $request->email;
                $shipping_info['address'] = $request->address;
                $shipping_info['country_id'] = $request->country_id;
                $shipping_info['state_id'] = $request->state_id;
                $shipping_info['city_id'] = $request->city_id;
                $shipping_info['postal_code'] = $request->postal_code; // This is pincode
                $shipping_info['phone'] = '+' . $request->country_code . $request->phone;
                $shipping_info['longitude'] = $request->longitude;
                $shipping_info['latitude'] = $request->latitude;
                $request->session()->put('guest_shipping_info', $shipping_info);
            }
            $carts = ($temp_user_id != null) ? Cart::where('temp_user_id', $temp_user_id)->get() : [];
        }

        if ($carts->isEmpty()) {
            flash(translate('Your cart is empty'))->warning();
            return redirect()->route('home');
        }

        $deliveryInfo = [];
        $pincodes = Pincode::where('city_id', $request->city_id)->where('is_active', 1)->get(); // Fetch pincodes for guest

        if ($auth_user != null) {
            $address = Address::where('id', $carts[0]['address_id'])->first();
            $deliveryInfo['country_id'] = $address->country_id;
            $deliveryInfo['city_id'] = $address->city_id;
            $pincodes = Pincode::where('city_id', $address->city_id)->where('is_active', 1)->get();
        } elseif ($temp_user_id != null) {
            $deliveryInfo['country_id'] = $request->country_id;
            $deliveryInfo['city_id'] = $request->city_id;
        }

        $carrier_list = array();
        if (get_setting('shipping_type') == 'carrier_wise_shipping') {
            $country_id = $auth_user != null ? $carts[0]['address']['country_id'] : $request->country_id;
            $zone = Country::where('id', $country_id)->first()->zone_id;

            $carrier_query = Carrier::where('status', 1);
            $carrier_query->whereIn('id', function ($query) use ($zone) {
                $query->select('carrier_id')->from('carrier_range_prices')
                    ->where('zone_id', $zone);
            })->orWhere('free_shipping', 1);
            $carrier_list = $carrier_query->get();
        }

        return view('frontend.delivery_info', compact('carts', 'carrier_list', 'deliveryInfo', 'pincodes'));
    }

    public function store_delivery_info(Request $request)
    {
        $request->validate([
            'payment_option' => 'required|string',
        ]);

        $authUser = auth()->user();
        $tempUser = $request->session()->has('temp_user_id') ? $request->session()->get('temp_user_id') : null;
        $carts = auth()->user() != null ?
            Cart::where('user_id', $authUser->id)->get() : ($tempUser != null ? Cart::where('temp_user_id', $request->session()->get('temp_user_id'))->get() : null);

        if ($carts->isEmpty()) {
            flash(translate('Your cart is empty'))->warning();
            return redirect()->route('home');
        }

        $shipping_info = $authUser != null ? Address::where('id', $carts[0]['address_id'])->first() : null;
        $deliveryInfo = [];

        if ($authUser != null) {
            $deliveryInfo['country_id'] = $shipping_info->country_id;
            $deliveryInfo['city_id'] = $shipping_info->city_id;
        } elseif ($tempUser != null) {
            $deliveryInfo['country_id'] = Session::get('guest_shipping_info')['country_id'];
            $deliveryInfo['city_id'] = Session::get('guest_shipping_info')['city_id'];
        }

        $total = 0;
        $tax = 0;
        $shipping = 0;
        $subtotal = 0;

        if ($carts && count($carts) > 0) {
            foreach ($carts as $key => $cartItem) {
                $product = Product::find($cartItem['product_id']);
                $tax += cart_product_tax($cartItem, $product, false) * $cartItem['quantity'];
                $subtotal += cart_product_price($cartItem, $product, false, false) * $cartItem['quantity'];

                if (get_setting('shipping_type') != 'carrier_wise_shipping' || $request['shipping_type_' . $product->user_id] == 'pickup_point') {
                    if ($request['shipping_type_' . $product->user_id] == 'pickup_point') {
                        $cartItem['shipping_type'] = 'pickup_point';
                        $cartItem['pickup_point'] = $request['pickup_point_id_' . $product->user_id];
                    } else {
                        $cartItem['shipping_type'] = 'home_delivery';
                    }
                    $cartItem['shipping_cost'] = 0;
                    if ($cartItem['shipping_type'] == 'home_delivery') {
                        $cartItem['shipping_cost'] = getShippingCost($carts, $key, $deliveryInfo);
                    }
                } else {
                    $cartItem['shipping_type'] = 'carrier';
                    $cartItem['carrier_id'] = $request['carrier_id_' . $product->user_id];
                    $cartItem['shipping_cost'] = getShippingCost($carts, $key, $deliveryInfo, $cartItem['carrier_id']);
                }

                $shipping += $cartItem['shipping_cost'];
                $cartItem->save();
            }
            $total = $subtotal + $tax + $shipping;

            return view('frontend.payment_select', compact('carts', 'shipping_info', 'total'));
        } else {
            flash(translate('Your Cart was empty'))->warning();
            return redirect()->route('home');
        }

        return response()->json(['success' => true]);
    }

    public function apply_coupon_code(Request $request)
    {
        $user = auth()->user();
        $temp_user = Session::has('temp_user_id') ? Session::get('temp_user_id') : null;
        $coupon = Coupon::where('code', $request->code)->first();
        $proceed = $request->proceed;
        $response_message = array();

        $canUseCoupon = true;
        if ($coupon && $coupon->type == 'welcome_base') {
            if ($user != null) {
                $userCoupon = $user->userCoupon;
                if (!$userCoupon) {
                    $canUseCoupon = false;
                }
            } else {
                $canUseCoupon = false;
            }
        }

        if ($coupon != null && $canUseCoupon) {
            if ($coupon->type != 'welcome_base') {
                $validationDateCheckCondition = strtotime(date('d-m-Y')) >= $coupon->start_date && strtotime(date('d-m-Y')) <= $coupon->end_date;
            } else {
                $validationDateCheckCondition = false;
                if ($userCoupon) {
                    $validationDateCheckCondition = $userCoupon->expiry_date >= strtotime(date('d-m-Y H:i:s'));
                }
            }
            if ($validationDateCheckCondition) {
                if (($user == null && Session::has('temp_user_id')) || CouponUsage::where('user_id', $user->id)->where('coupon_id', $coupon->id)->first() == null) {
                    $coupon_details = json_decode($coupon->details);

                    $user_carts = $user != null ?
                        Cart::where('user_id', $user->id)->where('owner_id', $coupon->user_id)->active()->get() :
                        Cart::where('owner_id', $coupon->user_id)->where('temp_user_id', $temp_user)->active()->get();

                    $coupon_discount = 0;

                    if ($coupon->type == 'cart_base' || $coupon->type == 'welcome_base') {
                        $subtotal = 0;
                        $tax = 0;
                        $shipping = 0;
                        foreach ($user_carts as $key => $cartItem) {
                            $product = Product::find($cartItem['product_id']);
                            $subtotal += cart_product_price($cartItem, $product, false, false) * $cartItem['quantity'];
                            $tax += cart_product_tax($cartItem, $product, false) * $cartItem['quantity'];
                            $shipping += $cartItem['shipping_cost'];
                        }
                        $sum = $subtotal + $tax + $shipping;
                        if ($coupon->type == 'cart_base' && $sum >= $coupon_details->min_buy) {
                            if ($coupon->discount_type == 'percent') {
                                $coupon_discount = ($sum * $coupon->discount) / 100;
                                if ($coupon_discount > $coupon_details->max_discount) {
                                    $coupon_discount = $coupon_details->max_discount;
                                }
                            } elseif ($coupon->discount_type == 'amount') {
                                $coupon_discount = $coupon->discount;
                            }
                        } elseif ($coupon->type == 'welcome_base' && $sum >= $userCoupon->min_buy) {
                            $coupon_discount = $userCoupon->discount_type == 'percent' ? (($sum * $userCoupon->discount) / 100) : $userCoupon->discount;
                        }

                        $user_carts->toQuery()->update([
                            'discount' => $coupon_discount / count($user_carts),
                            'coupon_code' => $request->code,
                            'coupon_applied' => 1
                        ]);
                    } elseif ($coupon->type == 'product_base') {
                        foreach ($user_carts as $key => $cartItem) {
                            $product = Product::find($cartItem['product_id']);
                            foreach ($coupon_details as $key => $coupon_detail) {
                                if ($coupon_detail->product_id == $cartItem['product_id']) {
                                    $discount_price_temp = 0;
                                    if ($coupon->discount_type == 'percent') {
                                        $discount_price_temp = (cart_product_price($cartItem, $product, false, false) * $coupon->discount / 100) * $cartItem['quantity'];
                                        $coupon_discount += $discount_price_temp;
                                    } elseif ($coupon->discount_type == 'amount') {
                                        $discount_price_temp = $coupon->discount * $cartItem['quantity'];
                                        $coupon_discount += $discount_price_temp;
                                    }
                                    Cart::where('id', $cartItem['id'])->update([
                                        'discount' => $discount_price_temp,
                                        'coupon_code' => $request->code,
                                        'coupon_applied' => 1
                                    ]);
                                }
                            }
                        }
                    }

                    if ($coupon_discount > 0) {
                        // $user_carts->toQuery()->update([
                        //     'discount' => $coupon_discount / count($user_carts),
                        //     'coupon_code' => $request->code,
                        //     'coupon_applied' => 1
                        // ]);

                        $response_message['response'] = 'success';
                        $response_message['message'] = translate('Coupon has been applied');
                    } else {
                        $response_message['response'] = 'warning';
                        $response_message['message'] = translate('This coupon is not applicable to your cart products!');
                    }
                } else {
                    $response_message['response'] = 'warning';
                    $response_message['message'] = translate('You already used this coupon!');
                }
            } else {
                $response_message['response'] = 'warning';
                $response_message['message'] = translate('Coupon expired!');
            }
        } else {
            $response_message['response'] = 'danger';
            $response_message['message'] = translate('Invalid coupon!');
        }

        if ($user != null) {
            $carts = Cart::where('user_id', $user->id)->active()->get();
        } else {
            $carts = ($temp_user != null) ? Cart::where('temp_user_id', $temp_user)->active()->get() : [];
        }


        if ($request->has('wallet_used_amount')) {
            $view_data = $request->only(['payment_method', 'wallet_used_amount']);
            $view_data['carts'] = $carts;
            return response()->json(array(
                'response_message' => $response_message,
                'html' => view('frontend.partials.cart.order-summary', $view_data)->render()
            ));
        }

        $returnHTML = view('frontend.partials.cart.cart_summary', compact('coupon', 'carts', 'proceed'))->render();
        return response()->json(array('response_message' => $response_message, 'html' => $returnHTML));
    }

    public function remove_coupon_code(Request $request)
    {
        $user = auth()->user();
        $temp_user = Session::has('temp_user_id') ? Session::get('temp_user_id') : null;
        $proceed = $request->proceed;
        $carts = $user != null ? Cart::where('user_id', $user->id) : Cart::where('temp_user_id', $temp_user);
        $status = $carts->update([
            'discount' => 0.00,
            'coupon_code' => '',
            'coupon_applied' => 0
        ]);
        $coupon = Coupon::where('code', $request->code)->first();
        $carts = $carts->active()->get();

        if ($request->has('wallet_used_amount')) {
            $response_message = [
                'response' => $status ? 'success' : 'warning',
                'message'  => $status ? 'Successfully removed discount coupon.' : 'Something went wrong!'
            ];
            $view_data = $request->only(['payment_method', 'wallet_used_amount']);
            $view_data['carts'] = $carts;
            return response()->json(array(
                'response_message' => $response_message,
                'html' => view('frontend.partials.cart.order-summary', $view_data)->render()
            ));
        }

        return view('frontend.partials.cart.cart_summary', compact('coupon', 'carts', 'proceed'));
    }

    public function order_confirmed()
    {
        // $combined_order = CombinedOrder::findOrFail(Session::get('combined_order_id'));
        if (!session()->has('payment_data')) {
            flash(translate('No payment data found.'))->error();
            return redirect()->route('home');
        }
        $combined_order = CombinedOrder::findOrFail(session('payment_data')['combined_order_id']);

        Session::forget('club_point');
        Session::forget('combined_order_id');
        Session::forget('payment_data'); // Clear payment data after confirmation

        foreach ($combined_order->orders as $order) {
            if ($order->notified == 0) {
                NotificationUtility::sendOrderPlacedNotification($order);
                $order->notified = 1;
                $order->save();
            }
        }
        Cart::where('user_id', auth()->id())->active()->delete(); // Clear cart after order confirmation

        return view('frontend.order_confirmed', compact('combined_order'));
    }

    public function guestCustomerInfoCheck(Request $request)
    {
        $user = addon_is_activated('otp_system') ?
            User::where('email', $request->email)->orWhere('phone', '+' . $request->phone)->first() :
            User::where('email', $request->email)->first();
        return ($user != null) ? true : false;
    }

    public function updateDeliveryAddress(Request $request)
    {
        $proceed = 0;
        $default_carrier_id = null;
        $default_shipping_type = 'home_delivery';
        $user = auth()->user();
        $shipping_info = array();

        $carts = $user != null ?
            Cart::where('user_id', $user->id)->active()->get() :
            Cart::where('temp_user_id', $request->session()->get('temp_user_id'))->active()->get();

        $carts->toQuery()->update(['address_id' => $request->address_id]);

        $country_id = $user != null ?
            Address::findOrFail($request->address_id)->country_id :
            $request->address_id;
        $city_id = $user != null ?
            Address::findOrFail($request->address_id)->city_id :
            $request->city_id;
        $shipping_info['country_id'] = $country_id;
        $shipping_info['city_id'] = $city_id;

        $carrier_list = array();
        if (get_setting('shipping_type') == 'carrier_wise_shipping') {
            $default_shipping_type = 'carrier';
            $zone = Country::where('id', $country_id)->first()->zone_id;

            $carrier_query = Carrier::where('status', 1);
            $carrier_query->whereIn('id', function ($query) use ($zone) {
                $query->select('carrier_id')->from('carrier_range_prices')
                    ->where('zone_id', $zone);
            })->orWhere('free_shipping', 1);
            $carrier_list = $carrier_query->get();

            if (count($carrier_list) > 1) {
                $default_carrier_id = $carrier_list->toQuery()->first()->id;
            }
        }

        $carts = $carts->fresh();

        foreach ($carts as $key => $cartItem) {
            if (get_setting('shipping_type') == 'carrier_wise_shipping') {
                $cartItem['shipping_cost'] = getShippingCost($carts, $key, $shipping_info, $default_carrier_id);
            } else {
                $cartItem['shipping_cost'] = getShippingCost($carts, $key, $shipping_info);
            }
            $cartItem['address_id'] = $user != null ? $request->address_id : 0;
            $cartItem['shipping_type'] = $default_shipping_type;
            $cartItem['carrier_id'] = $default_carrier_id;
            $cartItem->save();
        }

        $carts = $carts->fresh();

        return array(
            'delivery_info' => view('frontend.partials.cart.delivery_info', compact('carts', 'carrier_list', 'shipping_info'))->render(),
            'cart_summary' => view('frontend.partials.cart.cart_summary', compact('carts', 'proceed'))->render()
        );
    }

    public function updateDeliveryInfo(Request $request)
    {
        $proceed = 0;
        $user = auth()->user();
        $shipping_info = array();

        if ($user != null) {
            $carts = Cart::where('user_id', $user->id)->active()->get();
        } else {
            $temp_user_id = $request->session()->get('temp_user_id');
            $carts = ($temp_user_id != null) ? Cart::where('temp_user_id', $temp_user_id)->active()->get() : [];
        }

        $user_carts = $carts->toQuery()->where('owner_id', $request->user_id)->get();

        $country_id = $user != null ?
            Address::findOrFail($carts[0]->address_id)->country_id : $request->country_id;
        $city_id = $user != null ?
            Address::findOrFail($carts[0]->address_id)->city_id : $request->city_id;
        $shipping_info['country_id'] = $country_id;
        $shipping_info['city_id'] = $city_id;

        $shipping_type = $request->shipping_type;
        foreach ($user_carts as $key => $cartItem) {
            if ($shipping_type != 'carrier' || $shipping_type == 'pickup_point') {
                if ($shipping_type == 'pickup_point') {
                    $cartItem['shipping_type'] = 'pickup_point';
                    $cartItem['pickup_point'] = $request->type_id;
                } else {
                    $cartItem['shipping_type'] = 'home_delivery';
                }
                $cartItem['shipping_cost'] = 0;
                if ($cartItem['shipping_type'] == 'home_delivery') {
                    $cartItem['shipping_cost'] = getShippingCost($carts, $key, $shipping_info);
                }
            } else {
                $cartItem['shipping_type'] = 'carrier';
                $cartItem['carrier_id'] = $request->type_id;
                $cartItem['shipping_cost'] = getShippingCost($user_carts, $key, $shipping_info, $cartItem['carrier_id']);
            }

            $cartItem->save();
        }

        $carts = $carts->fresh();

        return view('frontend.partials.cart.cart_summary', compact('carts', 'proceed'))->render();
    }

    public function orderRePayment(Request $request)
    {
        $order = Order::findOrFail($request->order_id);
        if ($order) {
            $request->session()->put('payment_type', 'order_re_payment');
            $data['order_id'] = $order->id;
            $data['payment_method'] = $request->payment_option;
            $data['user_id'] = Auth::user()->id;
            $data['amount'] = $order->grand_total;
            $request->session()->put('payment_data', $data);

            if ($request->payment_option == 'phonepe') {
                $phonepeController = new ApiPhonepeController();
                $request->merge([
                    'payment_type' => $request->session()->get('payment_type')
                ]);
                return $phonepeController->pay($request);
            }

            $decorator = __NAMESPACE__ . '\\Payment\\' . str_replace(' ', '', ucwords(str_replace('_', ' ', $request->payment_option))) . "Controller";
            if (class_exists($decorator)) {
                return (new $decorator)->pay($request);
            } else {
                $manual_payment_data = array(
                    'name' => $request->payment_option,
                    'amount' => $order->grand_total,
                    'trx_id' => $request->trx_id,
                    'photo' => $request->photo
                );

                $order->payment_type = $request->payment_option;
                $order->manual_payment = 1;
                $order->manual_payment_data = json_encode($manual_payment_data);
                $order->save();

                flash(translate('Payment done.'))->success();
                return redirect()->route('purchase_history.details', encrypt($order->id));
            }
        }
        flash(translate('Order Not Found'))->warning();
        return back();
    }

    public function orderRePaymentDone($payment_data, $payment_details = null)
    {
        $order = Order::findOrFail($payment_data['order_id']);
        $order->payment_status = 'paid';
        $order->payment_details = $payment_details;
        $order->payment_type = $payment_data['payment_method'];
        $order->save();
        calculateCommissionAffilationClubPoint($order);

        if ($order->notified == 0) {
            NotificationUtility::sendOrderPlacedNotification($order);
            $order->notified = 1;
            $order->save();
        }

        Session::forget('payment_type');
        Session::forget('order_id');
        Session::forget('payment_data'); // Clear payment data after confirmation

        flash(translate('Payment done.'))->success();
        return redirect()->route('purchase_history.details', encrypt($order->id));
    }

    public function getPincodesByCity(Request $request)
    {
        $city_id = $request->input('city_id');
        $pincodes = Pincode::where('city_id', $city_id)->where('is_active', 1)->get(['pincode']);
        return response()->json(['pincodes' => $pincodes]);
    }

    public function deleteproductcheckout(Request $reqest, $ownerid, $productid)
    {
        try {
            DB::beginTransaction();
            Cart::where('owner_id', $ownerid)->where('product_id', $productid)->delete();
            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Product Removed Successfully From Cart'
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Unable to remove the product' . $e->getMessage()
            ]);
        }
    }
}
