<?php

namespace App\Http\Controllers;

use App\Http\Controllers\AffiliateController;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\Cart;
use App\Models\Address;
use App\Models\Country;
use App\Models\State;
use App\Models\City;
use App\Models\Pincode;
use App\Models\Product;
use App\Models\OrderDetail;
use App\Models\CouponUsage;
use App\Models\Coupon;
use App\Models\User;
use App\Models\CombinedOrder;
use App\Models\SmsTemplate;
use Illuminate\Support\Facades\Auth;
use Mail;
use App\Mail\InvoiceEmailManager;
use App\Models\OrdersExport;
use App\Models\ProductStock;
use App\Utility\NotificationUtility;
use CoreComponentRepository;
use App\Utility\SendSMSUtility;
use App\Utility\EmailUtility;
use Illuminate\Support\Facades\Route;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Notification;
use App\Notifications\OrderNotification;
use App\Services\OTP\MsgClub;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:view_all_orders|view_inhouse_orders|view_seller_orders|view_pickup_point_orders|view_all_offline_payment_orders'])->only('all_orders');
        $this->middleware(['permission:view_order_details'])->only('show');
        $this->middleware(['permission:delete_order'])->only('destroy', 'bulk_order_delete');
    }

    public function all_orders(Request $request)
    {
        // CoreComponentRepository::instantiateShopRepository();

        $date = $request->date;
        $sort_search = null;
        $delivery_status = null;
        $payment_status = '';
        $order_type = '';

        $orders = Order::orderBy('id', 'desc');
        $admin_user_id = get_admin()->id;

        if (Route::currentRouteName() == 'inhouse_orders.index' && Auth::user()->can('view_inhouse_orders')) {
            $orders = $orders->where('orders.seller_id', '=', $admin_user_id);
        } elseif (Route::currentRouteName() == 'seller_orders.index' && Auth::user()->can('view_seller_orders')) {
            $orders = $orders->where('orders.seller_id', '!=', $admin_user_id);
        } elseif (Route::currentRouteName() == 'pick_up_point.index' && Auth::user()->can('view_pickup_point_orders')) {
            if (get_setting('vendor_system_activation') != 1) {
                $orders = $orders->where('orders.seller_id', '=', $admin_user_id);
            }
            $orders->where('shipping_type', 'pickup_point')->orderBy('code', 'desc');
            if (Auth::user()->user_type == 'staff' && Auth::user()->staff->pick_up_point != null) {
                $orders->where('shipping_type', 'pickup_point')
                    ->where('pickup_point_id', Auth::user()->staff->pick_up_point->id);
            }
        } elseif (Route::currentRouteName() == 'all_orders.index' && Auth::user()->can('view_all_orders')) {
            if (get_setting('vendor_system_activation') != 1) {
                $orders = $orders->where('orders.seller_id', '=', $admin_user_id);
            }
        } elseif (Route::currentRouteName() == 'offline_payment_orders.index' && Auth::user()->can('view_all_offline_payment_orders')) {
            $orders = $orders->where('orders.manual_payment', 1);
            if ($request->order_type != null) {
                $order_type = $request->order_type;
                $orders = $order_type == 'inhouse_orders' ?
                    $orders->where('orders.seller_id', '=', $admin_user_id) :
                    $orders->where('orders.seller_id', '!=', $admin_user_id);
            }
        } elseif (Route::currentRouteName() == 'unpaid_orders.index' && Auth::user()->can('view_all_unpaid_orders')) {
            $orders = $orders->where('orders.payment_status', 'unpaid');
        } else {
            abort(403);
        }

        if ($request->search) {
            $sort_search = $request->search;
            $orders = $orders->where('code', 'like', '%' . $sort_search . '%');
        }
        if ($request->payment_status != null) {
            $orders = $orders->where('payment_status', $request->payment_status);
            $payment_status = $request->payment_status;
        }
        if ($request->delivery_status != null) {
            $orders = $orders->where('delivery_status', $request->delivery_status);
            $delivery_status = $request->delivery_status;
        }
        if ($date != null) {
            $orders = $orders->where('created_at', '>=', date('Y-m-d', strtotime(explode(" to ", $date)[0])) . ' 00:00:00')
                ->where('created_at', '<=', date('Y-m-d', strtotime(explode(" to ", $date)[1])) . ' 23:59:59');
        }
        $orders = $orders->paginate(15);
        $unpaid_order_payment_notification = get_notification_type('complete_unpaid_order_payment', 'type');
        return view('backend.sales.index', compact('orders', 'sort_search', 'order_type', 'payment_status', 'delivery_status', 'date', 'unpaid_order_payment_notification'));
    }

    public function show($id)
    {
        $order = Order::findOrFail(decrypt($id));
        $order_shipping_address = json_decode($order->shipping_address);
        $delivery_boys = User::where('city', $order_shipping_address->city)
            ->where('user_type', 'delivery_boy')
            ->get();

        if (env('DEMO_MODE') != 'On') {
            $order->viewed = 1;
            $order->save();
        }

        return view('backend.sales.show', compact('order', 'delivery_boys'));
    }

    public function create()
    {
        //
    }

    public function store(Request $request, $carts = null)
    {

        DB::beginTransaction();
        try {
            // dd($request->all());
            if (!$carts instanceof Cart) {
                $carts = Cart::with('product')->where('user_id', Auth::user()->id)->active()->get();
            }

            if ($carts->isEmpty()) {
                flash(translate('Your cart is empty'))->warning();
                return redirect()->route('home');
            }

            $address = $country_id = $state_id = $city_id = null;
            $address_id = $carts->first() ? $carts->first()->address_id : 0;
            if ($address_id) {
                $address = request_temp_data('address_' . $address_id);
                if (empty($address)) {
                    $address = Address::find($address_id);
                    request_temp_data('address_' . $address_id, $address);
                }
            }
            $country = $request->get('country', ($address ? $address->country->name : ""));
            if (is_numeric($country)) {
                $c_row = request_temp_data('country_' . $country);
                if (empty($c_row)) {
                    $c_row = Country::find($country);
                    request_temp_data('country_' . $country, $c_row);
                }
                $country_id = $c_row ? $c_row->id : "";
                $country = $c_row ? $c_row->name : "";
            }
            $state = $request->get('state', ($address ? $address->state->name : ""));
            if (is_numeric($state)) {
                $s_row = request_temp_data('state_' . $state);
                if (empty($s_row)) {
                    $s_row = State::find($state);
                    request_temp_data('state_' . $state, $s_row);
                }
                $state_id = $s_row ? $s_row->id : "";
                $state = $s_row ? $s_row->name : "";
            }
            $city = $request->get('city', ($address ? $address->city->name : ""));
            if (is_numeric($city)) {
                $ci_row = request_temp_data('city_' . $city);
                if (empty($ci_row)) {
                    $ci_row = City::find($city);
                    request_temp_data('city_' . $city, $ci_row);
                }
                $city_id = $ci_row ? $ci_row->id : "";
                $city = $ci_row ? $ci_row->name : "";
            }
            $postal_code = $request->get('postal_code', $request->get('pin_code', ($address ? $address->postal_code : "")));

            $lat = $request->get('latitude', $address ? $address->latitude : "");
            $lng = $request->get('longitude', $address ? $address->longitude : "");
            $shippingAddress = [
                'name'          => $request->get('name', Auth::user()->name),
                'email'         => $request->get('email', Auth::user()->email),
                'address'       => $request->get('address', $address ? $address->address : ""),
                'country'       => $country,
                'state'         => $state,
                'city'          => $city,
                'postal_code'   => $postal_code,
                'phone'         => format_mobile($request->get('phone', $request->get('mobile', $address ? $address->phone : ""))),
                'lat_lang'      => implode(",", array_filter([$lat, $lng]))
            ];

            // Store data as combined order
            $combined_order = new CombinedOrder;
            $combined_order->user_id = Auth::user()->id;
            $combined_order->shipping_address = json_encode($shippingAddress);
            if (!$combined_order->save()) {
                Log::error('Combined order is not created!');
                return false;
            }

            if (
                !Address::where('user_id', Auth::user()->id)
                    ->where('country_id', $country_id)
                    ->where('state_id', $state_id)
                    ->where('city_id', $city_id)
                    ->where('postal_code', $postal_code)
                    ->first()
            ) {
                $address = new Address;
                $address->user_id = Auth::user()->id;
                $address->order_id = $combined_order->id;
                $address->address = $shippingAddress['address'];
                $address->country_id = $country_id;
                $address->state_id = $state_id;
                $address->city_id = $city_id;
                $address->postal_code = $postal_code;
                $address->phone = $shippingAddress['phone'];
                $address->longitude = $lng;
                $address->latitude = $lat;
                $address->set_default = 1;
                $address->save();
            }

            $new_carts = [];
            foreach ($carts as $cart) {
                $user_id = $cart->product->user_id;
                if (!isset($new_carts[$user_id])) {
                    $new_carts[$user_id] = [];
                }
                $new_carts[$user_id][] = $cart;
            }

            foreach ($new_carts as $user_cart) {
                $order = new Order;
                $order->combined_order_id = $combined_order->id;
                $order->user_id = Auth::user()->id;
                $order->shipping_address = $combined_order->shipping_address;
                $order->additional_info = $request->get('additional_info');
                $order->payment_type = $request->payment_option;
                $order->delivery_viewed = '0';
                $order->payment_status_viewed = '0';
                $order->code = date('Ymd-His') . rand(10, 99);
                $order->date = strtotime('now');

                // Set initial payment status based on payment type
                $order->payment_status = $request->payment_option == 'cash_on_delivery' ? 'unpaid' : 'pending';
                $order->delivery_status = 'pending'; // Initial delivery status

                if ($request->payment_option == 'cash_on_delivery') {
                    $order->view_status = 1;
                }

                if ($order->save()) {

                    $total_discount = 0;
                    $sub_total = 0;
                    $tax = 0;
                    $shipping_cost = 0;
                    $discount_coupon = "";
                    foreach ($user_cart as $cart) {
                        if (empty($discount_coupon)) {
                            $discount_coupon = $cart->coupon_code;
                        }
                        $product = $cart->product;
                        $quantity = floatval($cart->quantity);
                        $product_variation_price = product_variation_discount($product, $cart->variation);

                        $total_discount += $product_variation_price['total'] * $quantity;
                        $cart_product_price = $product_variation_price['price'] * $quantity;
                        $sub_total += $cart_product_price;

                        $cart_tax = product_tax($product, $cart->variation) * $quantity;
                        $tax += $cart_tax;

                        $product_stock = request_temp_data('product_stock_' . $product->id);
                        if ($product->digital != 1 && $quantity > $product_stock->qty) {
                            flash(translate('The requested quantity is not available for ') . $product->getTranslation('name'))->warning();
                            throw new \Exception("The requested quantity is not available for " . $product->getTranslation('name'));
                        }

                        $order_detail = new OrderDetail;
                        $order_detail->order_id = $order->id;
                        $order_detail->seller_id = $product->user_id;
                        $order_detail->product_id = $product->id;
                        $order_detail->variation = $cart->variation;
                        $order_detail->price = $cart_product_price;
                        $order_detail->tax = $cart_tax;
                        $order_detail->shipping_type = $cart->shipping_type;
                        $order_detail->product_referral_code = $cart->product_referral_code;

                        $order->seller_id = $product->user_id;
                        $order->shipping_type = $cart->shipping_type;

                        if ($cart->shipping_type == 'pickup_point') {
                            $order->pickup_point_id = $cart->pickup_point;
                        }
                        if ($cart->shipping_type == 'carrier') {
                            $order->carrier_id = $cart->carrier_id;
                        }
                        $shipping_cost += floatval($cart->shipping_cost);

                        $order_detail->quantity = $quantity;

                        if (addon_is_activated('club_point')) {
                            $order_detail->earn_point = $product->earn_point;
                        }

                        if ($order_detail->save()) {
                            $product->num_of_sale += $quantity;
                            $product->save();

                            if ($product->added_by == 'seller' && $product->user->seller) {
                                $seller = $product->user->seller;
                                $seller->num_of_sale += $quantity;
                                $seller->save();
                            }

                            if (addon_is_activated('affiliate_system')) {
                                if ($order_detail->product_referral_code) {
                                    $referred_by_user = User::where('referral_code', $order_detail->product_referral_code)->first();
                                    if ($referred_by_user) {
                                        $affiliateController = new AffiliateController;
                                        $affiliateController->processAffiliateStats($referred_by_user->id, 0, $quantity, 0, 0);
                                    }
                                }
                            }
                        }

                        // if ($product->digital != 1) {
                        //     $product_stock->qty -= $quantity;
                        //     $product_stock->save();
                        // }
                    }

                    $order->grand_total = $sub_total + $tax + $shipping_cost - $total_discount;
                    $order->coupon_discount = $total_discount;
                    $userorgaddress = Address::with(['user', 'country', 'state', 'city'])
                        ->where('user_id', Auth::id())
                        ->orderBy('id')
                        ->first();

                    if ($order->save()) {
                        if ($request->payment_option == 'cash_on_delivery') {
                            $array = [
                                'view' => 'emails.invoice',
                                'subject' => translate('Order Placed - Order Code: ') . $order->code,
                                'from' => env('MAIL_FROM_ADDRESS'),
                                'order' => $order,
                                'userorgaddress' => $userorgaddress
                            ];
                            Mail::to($order->user->email)->send(new InvoiceEmailManager($array));
                        }

                        if ($total_discount) {
                            $coupon = Coupon::where('code', $discount_coupon)->first();
                            if ($coupon) {
                                $coupon_usage = new CouponUsage;
                                $coupon_usage->user_id = Auth::user()->id;
                                $coupon_usage->coupon_id = $coupon->id;
                                $coupon_usage->save();
                            }
                        }

                        $combined_order->grand_total += $order->grand_total;

                        if ($request->payment_option == 'cash_on_delivery') {
                            // sending the order pending status notification to the user on SMS & Email
                            Log::info("Attempting to send database email for pending order: Order {$order->code}");
                            EmailUtility::order_email($order, 'pending');
                            $this->sendHardcodedEmail($order, 'pending');
                        }
                    }
                }
            }
            $combined_order->save();

            $request->session()->put('combined_order_id', $combined_order->id);
            DB::commit();
            return $combined_order;
        } catch (Exception $e) {
            Log::error('Order placed error: ' . $e->getMessage());
            DB::rollBack();
            return false;
        }
    }

    public function edit($id)
    {
        //
    }

    public function update(Request $request, $id)
    {
        //
    }

    public function destroy($id)
    {

        DB::beginTransaction();
        try {
            $order = Order::findOrFail($id);
            if ($order) {
                $order->commissionHistory()->delete();
                foreach ($order->orderDetails as $key => $orderDetail) {
                    try {
                        product_restock($orderDetail);
                    } catch (\Exception $e) {
                    }
                    $orderDetail->delete();
                }

                // Send Email 1: Database-driven email
                Log::info("Attempting to send database email for order cancelled: Order {$order->code}");
                EmailUtility::order_email($order, 'cancelled');

                // Send Email 2: Hardcoded email
                $this->sendHardcodedEmail($order, 'cancelled');

                // // Send SMS notification if available
                // if (addon_is_activated('otp_system')) {
                //     $smsTemplate = SmsTemplate::where('identifier', 'order_cancelled_sms')->first();
                //     if ($smsTemplate && $smsTemplate->status == 1) {
                //         try {
                //             $phone = json_decode($order->shipping_address)->phone;
                //             $text = "Your order {$order->code} has been cancelled.";
                //             SendSMSUtility::sendSMS($phone, env('SMS_FROM_NUMBER', 'Store'), $text, null);
                //             Log::info("SMS sent for order cancelled: Order {$order->code}");
                //         } catch (\Exception $e) {
                //             Log::error("SMS failed for order cancelled: Order {$order->code} - " . $e->getMessage());
                //         }
                //     } else {
                //         Log::warning("SMS template 'order_cancelled_sms' not found or inactive for Order {$order->code}");
                //     }
                // }

                $order->delete();

                flash(translate('Order has been deleted successfully'))->success();
            } else {
                flash(translate('Something went wrong'))->error();
            }

            DB::commit(); // Save changes if successful
        } catch (\Exception $e) {
            DB::rollback(); // Undo changes if an error occurs
            Log::error($e->getMessage() . " for Order {$order->code}");
            flash(translate('Something went wrong'))->error();
        }
        return back();
    }

    public function bulk_order_delete(Request $request)
    {
        $ids = $request->get('id', []);
        $ids = is_array($ids) ? $ids : array_filter(explode(",", $ids));

        if (count($ids)) {
            foreach ($ids as $order_id) {
                $this->destroy($order_id);
            }
        }
        return 1;
    }

    public function order_details(Request $request)
    {
        $order = Order::findOrFail($request->order_id);
        $order->save();
        return view('seller.order_details_seller', compact('order'));
    }

    public function update_delivery_status(Request $request)
    {
        $order = Order::findOrFail($request->order_id);
        $order->delivery_viewed = '0';
        $order->delivery_status = $request->status;
        if ($request->status == 'delivered') {
            $order->delivered_date = date("Y-m-d H:i:s");
        }
        if ($order->save()) {
            Log::info("Delivery status updated to '{$request->status}' for Order {$order->code}");

            if (in_array($request->status, ['delivered', 'on_the_way', 'ready_for_pickup', 'picked_up', 'confirmed', 'cancelled'])) {
                $status_name = $request->status === "picked_up" ? 'ready_for_pickup' : $request->status;
                // Send Email 1: Database-driven email
                Log::info("Attempting to send database email for order {$status_name}: Order {$order->code}");
                // EmailUtility::order_email($order, $status_name);

                // Send Email 2: Hardcoded email
                $this->sendHardcodedEmail($order, $status_name);
                // return 'ooo';


                // if (addon_is_activated('otp_system') && in_array($status_name, ['delivered', 'on_the_way', 'ready_for_pickup', 'picked_up'])) {
                //     $msg_identifier = "order_delivered_sms";
                //     if($status_name === "on_the_way") {
                //         $msg_identifier = 'order_on_the_way_sms';
                //     } elseif($status_name === "ready_for_pickup") {
                //         $msg_identifier = 'product_pickup_sms';
                //     }
                //     $smsTemplate = SmsTemplate::where('identifier', $msg_identifier)->first();
                //     if ($smsTemplate && $smsTemplate->status == 1) {
                //         try {
                //             $phone = json_decode($order->shipping_address)->phone;
                //             $text = "Your order {$order->code} has been delivered.";
                //             // SendSMSUtility::sendSMS($phone, env('SMS_FROM_NUMBER', 'Store'), $text, null);
                //             Log::info("SMS sent for order delivered: Order {$order->code}");
                //         } catch (\Exception $e) {
                //             Log::error("SMS failed for order {$status_name}: Order {$order->code} - " . $e->getMessage());
                //         }
                //     } else {
                //         Log::warning("SMS template '{$msg_identifier}' not found or inactive for Order {$order->code}");
                //     }
                // }
            }

            if ($request->status == 'confirmed') {
                foreach ($order->orderDetails as $key => $orderDetail) {
                    $product_stock = ProductStock::where('product_id', $orderDetail->product_id)
                        ->where('variant', $orderDetail->variation)
                        ->first();
                    $product_stock->qty -= $orderDetail->quantity;
                    $product_stock->save();
                }
            }

            if ($request->status == 'cancelled' && $order->payment_type == 'wallet') {
                $user = User::where('id', $order->user_id)->first();
                $user->balance += $order->grand_total;
                $user->save();
            }

            if ($request->status == 'cancelled' && $order->user->user_type == 'seller' && $order->payment_status == 'paid' && $order->commission_calculated == 1) {
                $sellerEarning = $order->commissionHistory->seller_earning;
                $shop = $order->shop;
                $shop->admin_to_pay -= $sellerEarning;
                $shop->save();
            }

            if (Auth::user()->user_type == 'seller') {
                foreach ($order->orderDetails->where('seller_id', Auth::user()->id) as $key => $orderDetail) {
                    $orderDetail->delivery_status = $request->status;
                    $orderDetail->save();

                    if ($request->status == 'cancelled') {
                        product_restock($orderDetail);
                    }
                }
            } else {
                foreach ($order->orderDetails as $key => $orderDetail) {
                    $orderDetail->delivery_status = $request->status;
                    $orderDetail->save();

                    if ($request->status == 'cancelled') {
                        product_restock($orderDetail);
                    }

                    if (addon_is_activated('affiliate_system')) {
                        if (($request->status == 'delivered' || $request->status == 'cancelled') && $orderDetail->product_referral_code) {
                            $no_of_delivered = 0;
                            $no_of_canceled = 0;

                            if ($request->status == 'delivered') {
                                $no_of_delivered = $orderDetail->quantity;
                            }
                            if ($request->status == 'cancelled') {
                                $no_of_canceled = $orderDetail->quantity;
                            }

                            $referred_by_user = User::where('referral_code', $orderDetail->product_referral_code)->first();
                            $affiliateController = new AffiliateController;
                            $affiliateController->processAffiliateStats($referred_by_user->id, 0, 0, $no_of_delivered, $no_of_canceled);
                        }
                    }
                }
            }

            NotificationUtility::sendNotification($order, $request->status);

            if (get_setting('google_firebase') == 1 && $order->user->device_token != null) {
                $request->device_token = $order->user->device_token;
                $request->title = "Order updated!";
                $status = str_replace("_", "", $order->delivery_status);
                $request->text = "Your order {$order->code} has been {$status}";
                $request->type = "order";
                $request->id = $order->id;
                $request->user_id = $order->user->id;
                NotificationUtility::sendFirebaseNotification($request);
            }

            if (addon_is_activated('delivery_boy') && Auth::user()->user_type == 'delivery_boy') {
                $deliveryBoyController = new DeliveryBoyController;
                $deliveryBoyController->store_delivery_history($order);
            }

            return 1;
        }
        return 0;
    }

    public function update_tracking_code(Request $request)
    {
        $order = Order::findOrFail($request->order_id);
        $order->tracking_code = $request->tracking_code;
        $order->save();

        return 1;
    }

    public function update_payment_status(Request $request)
    {
        Log::info("update_payment_status called with status: {$request->status}, order_id: {$request->order_id}");

        $order = Order::findOrFail($request->order_id);
        try {
            $order->payment_status_viewed = '0';
            $order->payment_status = $request->status;
            if ($order->save()) {
                $status = Order::where('id', $request->order_id)->value('payment_status');
                // return $status;
                // Log the status update
                Log::info("Payment status updated to '{$status}' for Order {$order->code}");

                // Commission calculation for "paid" status
                // if ($order->payment_status == 'paid' && $order->commission_calculated == 0) {
                //     try {
                //         Log::info("Attempting commission calculation for Order {$order->code}");
                //         calculateCommissionAffilationClubPoint($order);
                //         Log::info("Commission calculated successfully for Order {$order->code}");
                //     } catch (\Exception $e) {
                //         Log::error("Commission calculation failed for Order {$order->code} - " . $e->getMessage());
                //     }
                // }

                $subject = "Your App - Payment Status Update: {$status}";
                $data = [
                    'order' => $order,
                    'subject' => $subject,
                    'header' => 'Payment Status Update',
                    'msg' => "Your order {$order->code} payment status has been updated to {$status}.",
                    'status' => $status,
                ];

                $this->sendHardcodedEmail($order, $status);
                Log::info("Email sent successfully for payment status '{$status}' to {$order->user->email} for Order {$order->code}");
                Log::info("SMS sent successfully for payment status '{$status}' to {$order->user->phone} for Order {$order->code}");
                return 1;
            }
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Failed to send email for payment status '{$status}' to {$order->user->email} for Order {$order->code} - " . $e->getMessage());
            return 0;
        }
    }

    public function assign_delivery_boy(Request $request)
    {
        if (addon_is_activated('delivery_boy')) {
            $order = Order::findOrFail($request->order_id);
            $order->assign_delivery_boy = $request->delivery_boy;
            $order->delivery_history_date = date("Y-m-d H:i:s");
            $order->save();

            $delivery_history = \App\Models\DeliveryHistory::where('order_id', $order->id)
                ->where('delivery_status', $order->delivery_status)
                ->first();

            if (empty($delivery_history)) {
                $delivery_history = new \App\Models\DeliveryHistory;
                $delivery_history->order_id = $order->id;
                $delivery_history->delivery_status = $order->delivery_status;
                $delivery_history->payment_type = $order->payment_type;
            }
            $delivery_history->delivery_boy_id = $request->delivery_boy;
            $delivery_history->save();

            if (env('MAIL_USERNAME') != null && get_setting('delivery_boy_mail_notification') == '1') {
                $array['view'] = 'emails.invoice';
                $array['subject'] = translate('You are assigned to deliver an order. Order code') . ' - ' . $order->code;
                $array['from'] = env('MAIL_FROM_ADDRESS');
                $array['order'] = $order;

                try {
                    Mail::to($order->delivery_boy->email)->queue(new InvoiceEmailManager($array));
                    Log::info("Delivery boy email queued for Order {$order->code}");
                } catch (\Exception $e) {
                    Log::error("Delivery boy email failed for Order {$order->code}: " . $e->getMessage());
                }
            }

            if (addon_is_activated('otp_system') && SmsTemplate::where('identifier', 'assign_delivery_boy')->first()->status == 1) {
                try {
                    SendSMSUtility::assign_delivery_boy($order->delivery_boy->phone, $order->code);
                    Log::info("SMS sent for delivery boy assignment: Order {$order->code}");
                } catch (\Exception $e) {
                    Log::error("SMS failed for delivery boy assignment: Order {$order->code} - " . $e->getMessage());
                }
            }
        }

        return 1;
    }

    public function orderBulkExport(Request $request)
    {
        if ($request->id) {
            return Excel::download(new OrdersExport($request->id), 'orders.xlsx');
        }
        return back();
    }

    public function unpaid_order_payment_notification_send(Request $request)
    {
        if ($request->order_ids != null) {
            $notificationType = get_notification_type('complete_unpaid_order_payment', 'type');
            foreach (explode(",", $request->order_ids) as $order_id) {
                $order = Order::where('id', $order_id)->first();
                $user = $order->user;
                if ($notificationType->status == 1 && $order->payment_status == 'unpaid') {
                    $order_notification['order_id'] = $order->id;
                    $order_notification['order_code'] = $order->code;
                    $order_notification['user_id'] = $order->user_id;
                    $order_notification['seller_id'] = $order->seller_id;
                    $order_notification['status'] = $order->payment_status;
                    $order_notification['notification_type_id'] = $notificationType->id;
                    Notification::send($user, new OrderNotification($order_notification));
                }
            }
            flash(translate('Notification Sent Successfully.'))->success();
        } else {
            flash(translate('Something went wrong!.'))->warning();
        }
        return back();
    }

    /**
     * Send hardcoded email for various order statuses
     */
    protected function sendHardcodedEmail($order, $status)
    {
        if (!$order->user) {
            Log::error('Invalid order user for sendHardcodedEmail function');
            return null;
        }
        $name = $order->user->name;
        $code = $order->code;
        $smsMessage = "Dear {$order->user->name}, your order {$order->code} has been " . str_replace(["_", "-"], " ", $status) . " on " . env('APP_NAME') . ".";
        switch ($status) {
            case 'confirmed':
                $subject = env('APP_NAME') . " - Order Confirmed";
                $header = translate('Great news!');
                $message = translate('Your order has been confirmed. Were preparing it for you.');
                $smsMessage = "Dear $name, your order $code is confirmed at Akshat Healthcare. Track here https://akshathealthcare.in/track-your-order. Thank you!";
                break;
            case 'cancelled':
                $subject = env('APP_NAME') . " - Order Cancellation";
                $header = translate('We are sorry to inform you');
                $message = translate('Your order has been cancelled. We apologize for any inconvenience.');
                $smsMessage = "Hi $name, your order $code, is cancelled as requested. Refund will process soon. - Akshat Healthcare";
                break;
            case 'delivered':
                $subject = env('APP_NAME') . " - Order Delivered";
                $header = translate('Good news!');
                $message = translate('Your order has been successfully delivered.');
                $smsMessage = "Dear $name, Your Order $code has been delivered! Thank you for Shopping with Akshat Healthcare!";
                break;
            case 'on_the_way':
                $subject = env('APP_NAME') . " - Order On The Way";
                $header = translate('Your order is on its way!');
                $message = translate('Your order is currently being shipped to you.');
                $smsMessage = "Hi $name, your Order $code, is on the way! Expect delivery soon. Track here https://akshathealthcare.in/track-your-order";
                break;
            case 'ready_for_pickup':
                $subject = env('APP_NAME') . " - Order Ready for Pickup";
                $header = translate('Ready for you!');
                $message = translate('Your order is ready for pickup at the designated location.');
                $smsMessage = "Dear $name, Your Order $code, has been picked up by our courier Partner. Track here https://akshathealthcare.in/track-your-order";
                break;
            case 'paid':
                $subject = env('APP_NAME') . " - Payment Confirmed";
                $header = translate('Thank you!');
                $message = translate('Payment for your order has been successfully confirmed.');
                $smsMessage = "Hi $name, Payment for Invoice $code is Received. Thank you for shopping at Akshat Healthcare!";
                break;
            case 'unpaid':
                $subject = env('APP_NAME') . " - Payment Pending";
                $header = translate('Action required');
                $message = translate('Your order is still unpaid. Please complete the payment soon.');
                $smsMessage = "Dear $name, Invoice $code is Unpaid. Pay Now at https://akshathealthcare.in/checkout to Proceed";
                break;
            case 'pending':
                $subject = env('APP_NAME') . " - Order Awaiting Confirmation";
                $header = translate('Your Order is Pending Confirmation');
                $message = translate('Your order has been placed but is awaiting confirmation. It will be processed soon.');
                $smsMessage = "Hi $name, Payment for order $code is pending. Kindly Complete here https://akshathealthcare.in/checkout - Akshat Healthcare";
                break;
            default:
                $subject = env('APP_NAME') . " - Order Update";
                $header = translate('Order update');
                $message = translate('Your order status has been updated.');
                break;
        }

        $data = [
            'order'     => $order,
            'subject'   => $subject,
            'header'    => $header,
            'msg'       => $message,
            'status'    => $status,
        ];

        $fromAddress = env('MAIL_FROM_ADDRESS', 'info@gitysoft.com');
        $fromName = env('MAIL_FROM_NAME', 'no-reply');

        // Send Email
        try {
            Mail::send('emails.order_notification', $data, function ($message) use ($order, $subject, $fromAddress, $fromName) {
                $message->to($order->user->email)
                    ->from($fromAddress, $fromName)
                    ->subject($subject);
            });
            Log::info("Hardcoded email sent successfully to {$order->user->email} for status: {$status}");
        } catch (\Exception $e) {
            Log::error("Failed to send hardcoded email for status {$status}: " . $e->getMessage());
        }

        $shipping_address = json_decode($order->shipping_address);

        $mobile = $shipping_address && isset($shipping_address->phone) && $shipping_address->phone ? $shipping_address->phone : $order->user->phone;
        $msg_club = new MsgClub();
        $msg_res = $msg_club->send($mobile, $smsMessage);
        if ($msg_res['success']) {
            Log::info("Additional SMS sent successfully to {$order->user->phone} for order {$status}: Order {$order->code}");
        } else {
            Log::error("Additional SMS API response error for order {$status}: Order {$order->code} - " . $msg_res['message']);
        }
    }
}
