<?php

namespace App\Services;

use Combinations;
use App\Models\Cart;
use App\Models\User;
use App\Models\Color;
use App\Models\Product;
use App\Models\Category;
use App\Models\Wishlist;
use Illuminate\Support\Str;
use App\Utility\ProductUtility;
use App\Utility\CategoryUtility;
use Illuminate\Support\Facades\DB;
use AizPackages\CombinationGenerate\Services\CombinationService;

class ProductService
{
    public function store(array $data)
    {
        $collection = collect($data);

        $approved = 1;
        if (auth()->user()->user_type == 'seller') {
            $user_id = auth()->user()->id;
            if (get_setting('product_approve_by_admin') == 1) {
                $approved = 0;
            }
        } else {
            $user_id = User::where('user_type', 'admin')->first()->id;
        }
        $tags = array();
        if ($collection['tags'][0] != null) {
            foreach (json_decode($collection['tags'][0]) as $key => $tag) {
                array_push($tags, $tag->value);
            }
        }
        $collection['tags'] = implode(',', $tags);
        $discount_start_date = null;
        $discount_end_date   = null;
        if ($collection['date_range'] != null) {
            $date_var               = explode(" to ", $collection['date_range']);
            $discount_start_date = strtotime($date_var[0]);
            $discount_end_date   = strtotime($date_var[1]);
        }
        unset($collection['date_range']);
        
        if ($collection['meta_title'] == null) {
            $collection['meta_title'] = $collection['name'];
        }
        if ($collection['meta_description'] == null) {
            $collection['meta_description'] = strip_tags($collection['description']);
        }

        if ($collection['meta_img'] == null) {
            $collection['meta_img'] = $collection['thumbnail_img'];
        }


        $shipping_cost = 0;
        if (isset($collection['shipping_type'])) {
            if ($collection['shipping_type'] == 'free') {
                $shipping_cost = 0;
            } elseif ($collection['shipping_type'] == 'flat_rate') {
                $shipping_cost = $collection['flat_shipping_cost'];
            }
        }
        unset($collection['flat_shipping_cost']);

        $slug = $collection['slug'] ?? Str::slug($collection['name']);
        $same_slug_count = Product::where('slug', 'LIKE', $slug . '%')->count();
        $slug_suffix = $same_slug_count ? '-' . ($same_slug_count + 1) : '';
        $collection['slug'] = $slug . $slug_suffix;

        $colors = json_encode(array());
        $old_colors = $collection->get('colors', []);
        $old_colors = is_array($old_colors) ? $old_colors : [];
        if (
            isset($collection['colors_active']) &&
            $collection['colors_active'] &&
            count($old_colors) > 0
        ) {
            $colors = json_encode($old_colors);
        }

        $options = ProductUtility::get_attribute_options($collection);

        $combinations = (new CombinationService())->generate_combination($options);
        
        if (count($combinations) > 0) {
            foreach ($combinations as $key => $combination) {
                $str = ProductUtility::get_combination_string($combination, $collection);

                unset($collection['price_' . str_replace('.', '_', $str)]);
                unset($collection['sku_' . str_replace('.', '_', $str)]);
                unset($collection['qty_' . str_replace('.', '_', $str)]);
                unset($collection['img_' . str_replace('.', '_', $str)]);
            }
        }

        unset($collection['colors_active']);

        $choice_options = array();
        $choice_nos = $collection->get('choice_no', []);
        $choice_nos = is_array($choice_nos) ? $choice_nos : [];
        
        $str = '';
        $item = array();
        foreach ($choice_nos as $key => $no) {
            $str = 'choice_options_' . $no;
            $item['attribute_id'] = $no;
            $attribute_data = array();
            $values = isset($collection[$str]) && is_array($collection[$str]) ? $collection[$str] : [];
            foreach ($values as $eachValue) {
                array_push($attribute_data, $eachValue);
            }
            unset($collection[$str]);

            $item['values'] = $attribute_data;
            array_push($choice_options, $item);
        }

        $choice_options = json_encode($choice_options, JSON_UNESCAPED_UNICODE);
        $attributes = json_encode($choice_nos);
        unset($collection['choice_no']);

        $published = 1;
        if ($collection['button'] == 'unpublish' || $collection['button'] == 'draft') {
            $published = 0;
        }
        unset($collection['button']);

        $collection['has_warranty'] = isset($collection['has_warranty']) ? 1 : 0;

        $data = $collection->merge(compact(
            'user_id',
            'approved',
            'discount_start_date',
            'discount_end_date',
            'shipping_cost',
            'slug',
            'colors',
            'choice_options',
            'attributes',
            'published'
        ))->toArray();
        
        return Product::create($data);
    }

    public function update(array $data, Product $product)
    {
        $collection = collect($data);

        $slug = $collection['slug'] ?? $product->slug ?? Str::slug($collection['name']);
        $same_slug_count = Product::where('slug', 'LIKE', $slug . '%')
            ->where('id', '!=', $product->id)
            ->count();
        $slug_suffix = $same_slug_count > 0 ? '-' . ($same_slug_count + 1) : '';
        $collection['slug'] = $slug . $slug_suffix;

        if(addon_is_activated('refund_request') && !isset($collection['refundable'])){
            $collection['refundable'] = 0;
        }

        if(!isset($collection['is_quantity_multiplied'])){
            $collection['is_quantity_multiplied'] = 0;
        }

        if(!isset($collection['cash_on_delivery'])){
            $collection['cash_on_delivery'] = 0;
        }
        if(!isset($collection['featured'])){
            $collection['featured'] = 0;
        }
        if(!isset($collection['todays_deal'])){
            $collection['todays_deal'] = 0;
        }


        $tags = array();
        if ($collection['tags'][0] != null) {
            foreach (json_decode($collection['tags'][0]) as $key => $tag) {
                array_push($tags, $tag->value);
            }
        }
        $collection['tags'] = implode(',', $tags);
        $discount_start_date = null;
        $discount_end_date   = null;
        if ($collection['date_range'] != null) {
            $date_var               = explode(" to ", $collection['date_range']);
            $discount_start_date = strtotime($date_var[0]);
            $discount_end_date   = strtotime($date_var[1]);
        }
        unset($collection['date_range']);
        
        if ($collection['meta_title'] == null) {
            $collection['meta_title'] = $collection['name'];
        }
        if ($collection['meta_description'] == null) {
            $collection['meta_description'] = strip_tags($collection['description']);
        }

        if ($collection['meta_img'] == null) {
            $collection['meta_img'] = $collection['thumbnail_img'];
        }

        if ($collection['lang'] != env("DEFAULT_LANGUAGE")) {
            unset($collection['name']);
            unset($collection['unit']);
            unset($collection['description']);
        }
        unset($collection['lang']);

        
        $shipping_cost = 0;
        if (isset($collection['shipping_type'])) {
            if ($collection['shipping_type'] == 'free') {
                $shipping_cost = 0;
            } elseif ($collection['shipping_type'] == 'flat_rate') {
                $shipping_cost = $collection['flat_shipping_cost'];
            }
        }
        unset($collection['flat_shipping_cost']);

        $colors = json_encode(array());
        if (
            isset($collection['colors_active']) && 
            $collection['colors_active'] &&
            $collection['colors'] &&
            count($collection['colors']) > 0
        ) {
            $colors = json_encode($collection['colors']);
        }

        $options = ProductUtility::get_attribute_options($collection);

        $combinations = (new CombinationService())->generate_combination($options);
        if (count($combinations) > 0) {
            foreach ($combinations as $key => $combination) {
                $str = ProductUtility::get_combination_string($combination, $collection);

                unset($collection['price_' . str_replace('.', '_', $str)]);
                unset($collection['sku_' . str_replace('.', '_', $str)]);
                unset($collection['qty_' . str_replace('.', '_', $str)]);
                unset($collection['img_' . str_replace('.', '_', $str)]);
            }
        }

        unset($collection['colors_active']);

        $choice_options = array();
        if (isset($collection['choice_no']) && $collection['choice_no']) {
            $str = '';
            $item = array();
            foreach ($collection['choice_no'] as $key => $no) {
                $str = 'choice_options_' . $no;
                $item['attribute_id'] = $no;
                $attribute_data = array();
                // foreach (json_decode($request[$str][0]) as $key => $eachValue) {
                $eachValues = isset($collection[$str]) && is_array($collection[$str]) ? $collection[$str] : [];
                foreach ($eachValues as $key => $eachValue) {
                    // array_push($data, $eachValue->value);
                    array_push($attribute_data, $eachValue);
                }
                unset($collection[$str]);

                $item['values'] = $attribute_data;
                array_push($choice_options, $item);
            }
        }

        $choice_options = json_encode($choice_options, JSON_UNESCAPED_UNICODE);

        if (isset($collection['choice_no']) && $collection['choice_no']) {
            $attributes = json_encode($collection['choice_no']);
            unset($collection['choice_no']);
        } else {
            $attributes = json_encode(array());
        }

        $collection['has_warranty'] = isset($collection['has_warranty']) ? 1 : 0;
        
        unset($collection['button']);
        
        $data = $collection->merge(compact(
            'discount_start_date',
            'discount_end_date',
            'shipping_cost',
            'slug',
            'colors',
            'choice_options',
            'attributes'
        ))->toArray();
        
        $product->update($data);

        return $product;
    }
    
    public function product_duplicate_store($product)
    {
        $product_new = $product->replicate();
        $product_new->slug = $product_new->slug . '-' . Str::random(5);
        $product_new->approved = (get_setting('product_approve_by_admin') == 1 && $product->added_by != 'admin') ? 0 : 1;
        $product_new->save();

        return $product_new;
    }

    public function destroy($id)
    {
        $product = Product::findOrFail($id);
        $product->product_translations()->delete();
        $product->categories()->detach();
        $product->stocks()->delete();
        $product->taxes()->delete();
        $product->wishlists()->delete();
        $product->carts()->delete();
        $product->frequently_bought_products()->delete();
        $product->last_viewed_products()->delete();
        $product->flash_deal_products()->delete();
        deleteProductReview($product);
        Product::destroy($id);
    }

    public function product_search(array $data)
    {   
        $collection     = collect($data);
        $auth_user      = auth()->user();
        $productType    = $collection['product_type'];
        $products       = Product::query();
    
        if($collection['category'] != null ) {
            $category = Category::with('childrenCategories')->find($collection['category']);
            $products = $category->products();
        }
        
        $products = in_array($auth_user->user_type, ['admin', 'staff']) ? $products->where('products.added_by', 'admin') : $products->where('products.user_id', $auth_user->id);
        $products->where('published', '1')->where('auction_product', 0)->where('approved', '1');

        if($productType == 'physical'){
            $products->where('digital', 0)->where('wholesale_product', 0);
        }
        elseif($productType == 'digital'){
            $products->where('digital', 1);
        }
        elseif($productType == 'wholesale'){
            $products->where('wholesale_product', 1);
        }

        if($collection['product_id'] != null){
            $products->where('id', '!=' , $collection['product_id']);
        }
        
        if ($collection['search_key'] != null) {
            $products->where('name','like', '%' . $collection['search_key'] . '%');
        }    

        return $products->limit(20)->get();
    }

    public function setCategoryWiseDiscount(array $data)
    {
        $auth_user      = auth()->user();
        $discount_start_date = null;
        $discount_end_date   = null;
        if ($data['date_range'] != null) {
            $date_var               = explode(" to ", $data['date_range']);
            $discount_start_date = strtotime($date_var[0]);
            $discount_end_date   = strtotime($date_var[1]);
        }
        $seller_product_discount =  isset($data['seller_product_discount']) ? $data['seller_product_discount'] : null ;
        $admin_id = User::where('user_type', 'admin')->first()->id;

        $products = Product::where('category_id', $data['category_id'])->where('auction_product', 0);
        if(in_array($auth_user->user_type, ['admin', 'staff']) && $seller_product_discount == 0){
            $products = $products->where('user_id', $admin_id);
        }
        elseif($auth_user->user_type == 'seller'){
            $products = $products->where('user_id', $auth_user->id);
        }

        $products->update([
            'discount' => $data['discount'],
            'discount_type' => 'percent',
            'discount_start_date' => $discount_start_date,
            'discount_end_date' => $discount_end_date,
        ]);
        return 1;
    }
    
    /**
     * Get substitute products based on predefined rules.
     */
    // public static function getSubstitutes($productId, $check_stock = false)
    // {
    //     $product = Product::select('id', 'category_id', 'brand_id', 'tags', 'current_stock', 'unit_price')
    //         ->find($productId);

    //     if($check_stock) {
    //         if (!$product || $product->current_stock > 0) {
    //             return [];
    //         }
    //     }
        
    //     $wholesale_addon = addon_is_activated('wholesale');
    //     $verified_sellers = verified_sellers_id();

    //     // Step 1: Get related category IDs (including subcategories)
    //     $relatedCategoryIds = CategoryUtility::children_ids($product->category_id);
    //     $relatedCategoryIds = is_array($relatedCategoryIds) && count($relatedCategoryIds) ? $relatedCategoryIds : [];
    //     $relatedCategoryIds[] = $product->category_id;

    //     // Step 2: Find substitutes within the same category or subcategory
    //     $substitutes = Product::with('brand')
    //         ->whereIn('category_id', $relatedCategoryIds)
    //         ->where('id', '!=', $product->id)
    //         // ->where('current_stock', '>', 0);
    //         ->isApprovedPublished()
    //         ->where('auction_product', 0);
    //     if (!$wholesale_addon) {
    //         $substitutes = $substitutes->where('wholesale_product', 0);
    //     }
    //     if (get_setting('vendor_system_activation') == 1) {
    //         $substitutes = $substitutes->where(function ($p) use ($verified_sellers) {
    //             $p->where('added_by', 'admin')->orWhere(function ($q) use ($verified_sellers) {
    //                 $q->whereIn('user_id', $verified_sellers);
    //             });
    //         });
    //     } else {
    //         $substitutes = $substitutes->where('added_by', 'admin');
    //     }
    //     $substitutes = $substitutes->orderBy('unit_price', 'asc')
    //         ->limit(5)
    //         ->get();
        
            

    //     // Step 3: If not enough, find substitutes from the same brand
    //     if ($substitutes->isEmpty() && $product->brand_id) {
    //         $brandSubstitutes = Product::with('brand')
    //             ->where('brand_id', $product->brand_id)
    //             ->where('id', '!=', $product->id)
    //             // ->where('current_stock', '>', 0)
    //             ->isApprovedPublished()
    //             ->where('auction_product', 0);
    //         if (!$wholesale_addon) {
    //             $brandSubstitutes = $brandSubstitutes->where('wholesale_product', 0);
    //         }
    //         if (get_setting('vendor_system_activation') == 1) {
    //             $brandSubstitutes = $brandSubstitutes->where(function ($p) use ($verified_sellers) {
    //                 $p->where('added_by', 'admin')->orWhere(function ($q) use ($verified_sellers) {
    //                     $q->whereIn('user_id', $verified_sellers);
    //                 });
    //             });
    //         } else {
    //             $brandSubstitutes = $brandSubstitutes->where('added_by', 'admin');
    //         }
    //         $brandSubstitutes = $brandSubstitutes->limit(5 - $substitutes->count())
    //             ->get();
            
    //         $substitutes = $substitutes->merge($brandSubstitutes);
    //     }

    //     // Step 4: If still not enough, find substitutes based on tags (keywords)
    //     if ($substitutes->isEmpty() && !empty($product->tags)) {
    //         $tags = explode(',', $product->tags);
    //         $tagSubstitutes = Product::with('brand')
    //             ->where('id', '!=', $product->id)
    //             // ->where('current_stock', '>', 0)
    //             ->where(function ($query) use ($tags) {
    //                 foreach ($tags as $tag) {
    //                     $query->orWhere('tags', 'LIKE', "%$tag%");
    //                 }
    //             })
    //             ->isApprovedPublished()
    //             ->where('auction_product', 0);
    //         if (!$wholesale_addon) {
    //             $tagSubstitutes = $tagSubstitutes->where('wholesale_product', 0);
    //         }
    //         if (get_setting('vendor_system_activation') == 1) {
    //             $tagSubstitutes = $tagSubstitutes->where(function ($p) use ($verified_sellers) {
    //                 $p->where('added_by', 'admin')->orWhere(function ($q) use ($verified_sellers) {
    //                     $q->whereIn('user_id', $verified_sellers);
    //                 });
    //             });
    //         } else {
    //             $tagSubstitutes = $tagSubstitutes->where('added_by', 'admin');
    //         }
    //         $tagSubstitutes = $tagSubstitutes->limit(5 - $substitutes->count())
    //             ->get();
            
    //         $substitutes = $substitutes->merge($tagSubstitutes);
    //     }

    //     return $substitutes;
    // }
    public static function getSubstitutes($productId, $check_stock = true, $limit = 5)
    {
        try {
            $product = Product::select('id', 'category_id', 'brand_id', 'tags', 'unit_price')
                ->find($productId);
        
            if (!$product) {
                return [];
            }
        
            $query = Product::with('brand')
                ->where('id', '!=', $product->id)
                ->isApprovedPublished()
                ->whereNotIn('slug', ['N/A', 'n/a', ""])
                ->where('auction_product', 0)
                ->whereExists(function ($q) {
                    $q->select(DB::raw(1))
                      ->from('product_stocks')
                      ->whereRaw('product_stocks.product_id = products.id')
                      ->groupBy('product_stocks.product_id')
                      ->havingRaw('SUM(product_stocks.qty) > 0');
                });
                
            $wholesale_addon = addon_is_activated('wholesale');
            if (!$wholesale_addon) {
                $query->where('wholesale_product', 0);
            }
        
            // **Step 1: Find Substitutes in Same Category and Same Generic Name**
            $substitutes = (clone $query)
                ->where(function ($query) use ($product) {
                    $tags = explode(',', $product->tags);
                    $tags = array_filter($tags);
                    foreach ($tags as $tag) {
                        $query->orWhere('tags', 'LIKE', "%$tag%");
                    }
                    $query->orWhereBetween('unit_price', [$product->unit_price * 0.8, $product->unit_price * 1.2]) // Price Range 80% - 120%
                        ->where('category_id', $product->category_id);
                    if($product->brand_id) {
                        $query->orWhere('brand_id', $product->brand_id);
                    }
                })
                ->orderBy('unit_price', 'asc')
                ->limit($limit)
                ->get();
                
            return $substitutes;
        } catch(Exception $e) {
            Log::error('Error fetching substitute products: ' . $e->getMessage());
            return [];
        }
    }

}